/*
 * Copyright (C) 2010, Panasonic Corporation.
 *                       All Rights Reserved.
 *
 */

#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/usb.h>
#include <linux/usb/quirks.h>
#include "usb.h"
#include "hcd.h"
#include "../host/ehci-test.h"

#ifdef	CONFIG_USB_SUSPEND

static void recursiveSuspend( struct usb_device* udev )
{
	unsigned	port;
	int ret;
	
	for( port = 1; port <= udev->maxchild ; port++ ){
		struct usb_device	*tmp;
		
		tmp = udev->children[ port-1 ];
		if( tmp )
			recursiveSuspend( tmp );
	}
	
	if( !udev->dev.type || !udev->dev.type->pm || !udev->dev.type->pm->suspend ){
		printk( "USB-device(%s) don't have suspend.\n", udev->devpath );
		return;
	}
	ret = udev->dev.type->pm->suspend( &udev->dev );
	printk( "suspend device(%s):ret=%d\n",udev->devpath, ret );
}

static void recursiveResume( struct usb_device* udev )
{
	unsigned	port;
	int ret;
	
	if( !udev->dev.type || !udev->dev.type->pm || !udev->dev.type->pm->resume ){
		printk( "USB-device(%s) don't have resume.\n", udev->devpath );
		return;
	}
	ret = udev->dev.type->pm->resume( &udev->dev );
	printk( "resume device(%s):ret=%d\n",udev->devpath, ret );
	
	for( port = 1; port <= udev->maxchild ; port++ ){
		struct usb_device	*tmp;
		
		tmp = udev->children[ port-1 ];
		if( tmp )
			recursiveResume( tmp );
	}
}

static void do_susrum( struct usb_device* udev )
{
	printk( "USB TEST START [SUSPEND AND RESUME] udev->devpath = %s\n", udev->devpath );
	
	printk( "sleep 15sec.\n" ); msleep(15000);
	
	recursiveSuspend( udev );
	
	printk( "sleep 15sec.\n" ); msleep(15000);
	
	recursiveResume( udev );
	
	printk( "USB TEST END \n" );
}
#else
static void do_susrum( struct usb_device* udev )
{
	printk( "please enable CONFIG_USB_SUSPEND.\n" );
}
#endif

static void print_device_descriptor( const char* buf )
{
	printk("      DEVICE DESCRIPTOR = \n");
	printk("         bLength            = 0x%02x \n", *buf  );
	printk("         bDescriptorType    = 0x%02x \n", *(buf+1)  );
	printk("         bcdUSB             = 0x%02x%02x \n", *(buf+3), *(buf+2)  );
	printk("         bDeviceClass       = 0x%02x \n", *(buf+4)  );
	printk("         bDeviceSubClass    = 0x%02x \n", *(buf+5)  );
	printk("         bDeviceProtocol    = 0x%02x \n", *(buf+6)  );
	printk("         bMaxPacketSize0    = 0x%02x \n", *(buf+7)  );
	printk("         idVendor           = 0x%02x%02x \n", *(buf+9), *(buf+8)  );
	printk("         idProduct          = 0x%02x%02x \n", *(buf+11), *(buf+10)  );
	printk("         bcdDevice          = 0x%02x%02x \n", *(buf+13), *(buf+12)  );
	printk("         iManufacturer      = 0x%02x \n", *(buf+14)  );
	printk("         iProduct           = 0x%02x \n", *(buf+15)  );
	printk("         iSerialNumber      = 0x%02x \n", *(buf+16)  );
	printk("         bNumConfigurations = 0x%02x \n", *(buf+17)  );
}

static void print_config_descriptor( const char* buf )
{
	printk("      CONFIG DESCRIPTOR = \n");
	printk("         bLength              = 0x%02x \n", *buf  );
	printk("         bDescriptorType      = 0x%02x \n", *(buf+1)  );
	printk("         wTotalLength         = 0x%02x%02x \n", *(buf+3), *(buf+2)  );
	printk("         bNumInterfaces       = 0x%02x \n", *(buf+4)  );
	printk("         bConfigurationValue  = 0x%02x \n", *(buf+5)  );
	printk("         iConfiguration       = 0x%02x \n", *(buf+6)  );
	printk("         bmAttributes         = 0x%02x \n", *(buf+7)  );
	printk("         MaxPower             = 0x%02x \n", *(buf+8)  );
}

static void do_getdev( struct usb_device* udev )
{
	int status = -ETIMEDOUT;
	u8* buf;
	
	buf = (u8*)kzalloc( 256, GFP_KERNEL );
	
	printk( "USB TEST START [SINGLE_STEP_GET_DEV_DESCRIPTOR] udev->devpath = %s\n", udev->devpath );
	
	printk( "USB TEST (15sec SOF) \n" );
	msleep(15000);

	printk( "USB TEST (Get Device Descriptor) \n" );
	status = usb_control_msg(
				udev,
				usb_rcvctrlpipe(udev, 0),
				USB_REQ_GET_DESCRIPTOR,
				USB_DIR_IN,
				0x100,
				0,
				buf,
				256,
				HZ * USB_CTRL_GET_TIMEOUT
			);
	
	if( status < 0 ){
		printk( "Internal USB Host Test Driver Error (Get Device Descriptor Error) \n" );
		kfree(buf);
		return;
	}

	print_device_descriptor( buf );
	
	printk( "USB TEST END \n" );
	
	kfree( buf );
}

static void do_setfea( struct usb_device* udev )
{
	int status = -ETIMEDOUT;
	u8* buf;
	
	printk( "USB TEST START [SINGLE_STEP_SET_FEATURE] udev->devpath = %s\n", udev->devpath );

	if( udev->test_usb_wait_qtd ){
		printk( "The test is being done by someone!\n");
		return;
	}
	
	if( !udev->parent ){
		printk( "This [SINGLE_STEP_SET_FEATURE] test is not for root-hub.\n");
		return;
	}
	
	buf = (u8*)kzalloc( 256, GFP_KERNEL );
	
	udev->test_usb_wait_qtd = (__le32*)0xFFFFFFFF;

	printk( "USB TEST (Get Config Descriptor Setup-stage) \n" );
	printk( "USB TEST (15sec wait) \n" );
	status = usb_control_msg(
				udev,
				usb_rcvctrlpipe(udev, 0),
				USB_REQ_GET_DESCRIPTOR,
				USB_DIR_IN,
				0x200,
				0,
				buf,
				256,
				HZ * USB_CTRL_GET_TIMEOUT
			);
	
	if( status < 0 ){

		printk( "Internal USB Host Test Driver Error (Get Config Descriptor Error) \n" );
		kfree(buf);
		return;
	}
	
	printk( "USB TEST (Get Config Descriptor Data-stage) \n" );
	printk( "USB TEST (Get Config Descriptor Status-stage) \n" );
	
	print_config_descriptor( buf );
	
	printk( "USB TEST END \n" );
	
	kfree( buf );
}

static int _test_mode( struct usb_device* udev, u32 testid, enum usb_device_speed speed )
{
	struct usb_hcd* hcd;
	int ret;
	
	ret = 0;
	
	if( !udev->bus || !udev->bus->root_hub || !udev->bus->root_hub->bus )
		return -EINVAL;
	hcd = bus_to_hcd( udev->bus->root_hub->bus );
	if(hcd == NULL)
		return -EINVAL;
	
	switch( testid ){
		case USB_TESTMD_PID_SUSRUM:
			do_susrum( udev );
			break;
		
		case USB_TESTMD_PID_GETDEV:
			do_getdev( udev );
			break;
		
		case USB_TESTMD_PID_SETFEA:
			do_setfea( udev );
			break;
		
		default:
			ret = usb_ehci_panasonic_test_mode( hcd, testid, speed );
			break;
	}
	
	return ret;
}

static int test_mode( struct usb_device* udev, u32 testid, enum usb_device_speed speed )
{
	int ret;
	
	usb_lock_device( udev );
	{
		ret = _test_mode( udev, testid, speed );
	}
	usb_unlock_device( udev );
	
	return ret;
}

static int internal_host_test_probe( struct usb_interface *intf, const struct usb_device_id *id )
{
	struct usb_device *udev;
	u32 testid;
	int ret;
	
	udev = interface_to_usbdev( intf );
	if( !udev || !udev->bus || !udev->parent )
		return  -EINVAL;

	if( udev->parent->parent ){
		printk( "Internal USB Host Test Driver Error (Test Device must attach the RootHub) \n" );
		return -EINVAL;
	}
	
#ifdef CONFIG_USB_PANASONIC_USE_DUMMY_TEST_DEVICES
	testid = dummyid2testid( id->idVendor, id->idProduct );
#else
	testid = id->idProduct;
#endif
	
	printk( "%s speed Test Device ( Vendor %04x : Product %04x : TestID %04x ) Attached\n",
		(udev->speed==USB_SPEED_LOW)?  "low":
		(udev->speed==USB_SPEED_FULL)? "full":
		(udev->speed==USB_SPEED_HIGH)? "high":"?",
		id->idVendor, id->idProduct, testid );
	
	if( testid == USB_TESTMD_PID_SUSRUM ){
		udev = udev->parent;
	}
	ret = _test_mode( udev, testid, udev->speed );
	
	printk( "Internal USB Host Test Driver END (ret = %d) \n", ret );
	
	return -1;
}

static struct usb_device_id internal_host_test_id_table [] = {
	{USB_DEVICE (USB_TESTMD_VDR_SE0,	USB_TESTMD_PID_SE0)		},
	{USB_DEVICE (USB_TESTMD_VDR_J,		USB_TESTMD_PID_J)		},
	{USB_DEVICE (USB_TESTMD_VDR_K,		USB_TESTMD_PID_K)		},
	{USB_DEVICE (USB_TESTMD_VDR_PACKET,	USB_TESTMD_PID_PACKET)	},
	{USB_DEVICE (USB_TESTMD_VDR_SUSRUM,	USB_TESTMD_PID_SUSRUM)	},
	{USB_DEVICE (USB_TESTMD_VDR_GETDEV,	USB_TESTMD_PID_GETDEV)	},
	{USB_DEVICE (USB_TESTMD_VDR_SETFEA,	USB_TESTMD_PID_SETFEA)	},
#ifdef CONFIG_USB_PANASONIC_USE_DUMMY_TEST_DEVICES
	USB_DUMMY_TEST_DEVICES_ID_TABLE
#endif
	{}
};
MODULE_DEVICE_TABLE (usb, internal_host_test_id_table);

static struct usb_driver usbtest_driver = {
	.name =		"usb_host_test_driver",
	.id_table =	internal_host_test_id_table,
	.probe =	internal_host_test_probe,
};

static int __init usb_test_mode_init( void )
{
	int ret;
	
	ret = usb_register( &usbtest_driver );
	usb_set_host_test_function( test_mode );
	return ret;
}
module_init (usb_test_mode_init);

static void __exit usb_test_mode_exit( void )
{
	usb_set_host_test_function( NULL );
	usb_deregister( &usbtest_driver );
}
module_exit(usb_test_mode_exit);
MODULE_LICENSE ("GPL");

